-- ------------------------------------------------------------------------------ --
--                                TradeSkillMaster                                --
--                http://www.curse.com/addons/wow/tradeskill-master               --
--                                                                                --
--             A TradeSkillMaster Addon (http://tradeskillmaster.com)             --
--    All Rights Reserved* - Detailed license information included with addon.    --
-- ------------------------------------------------------------------------------ --

local _, TSM = ...
local Professions = TSM.MainUI.Settings:NewPackage("Professions")
local L = LibStub("AceLocale-3.0"):GetLocale("TradeSkillMaster") -- loads the localization table
local private = { altCharacters = {}, altGuilds = {} }



-- ============================================================================
-- Module Functions
-- ============================================================================

function Professions.OnInitialize()
	TSM.MainUI.Settings.RegisterSettingPage("Professions", "middle", private.GetProfessionsSettingsFrame)
end



-- ============================================================================
-- Professions Settings UI
-- ============================================================================

function private.GetProfessionsSettingsFrame()
	wipe(private.altCharacters)
	wipe(private.altGuilds)
	for name in pairs(TSMAPI_FOUR.PlayerInfo.GetCharacters(true)) do
		tinsert(private.altCharacters, name)
	end
	for name in pairs(TSMAPI_FOUR.PlayerInfo.GetGuilds()) do
		tinsert(private.altGuilds, name)
	end

	return TSMAPI_FOUR.UI.NewElement("ScrollFrame", "professionSettings")
		:SetStyle("padding", 10)
		:AddChild(TSM.MainUI.Settings.CreateHeadingLine("inventoryTitle", L["Inventory Options"]))
		:AddChild(TSM.MainUI.Settings.CreateSettingLine("ignoreChar", L["Characters to ignore:"])
			:AddChild(TSMAPI_FOUR.UI.NewElement("Dropdown", "charDropdown")
				:SetMultiselect(true)
				:SetItems(private.altCharacters, TSM.db.global.craftingOptions.ignoreCharacters)
				:SetScript("OnSelectionChanged", private.IgnoreCharactersOnSelectionChanged)
			)
		)
		:AddChild(TSM.MainUI.Settings.CreateSettingLine("ignoreGuild", L["Guilds to ignore:"])
			:AddChild(TSMAPI_FOUR.UI.NewElement("Dropdown", "guildDropdown")
				:SetMultiselect(true)
				:SetItems(private.altGuilds, TSM.db.global.craftingOptions.ignoreGuilds)
				:SetScript("OnSelectionChanged", private.IgnoreGuildsOnSelectionChanged)
			)
		)
		:AddChild(TSM.MainUI.Settings.CreateHeadingLine("priceTitle", L["Default Price Configuration"]))
		:AddChild(TSMAPI_FOUR.UI.NewElement("Frame", "matCost")
			:SetLayout("HORIZONTAL")
			:SetStyle("height", 26)
			:SetStyle("margin", { bottom = 8 })
			:AddChild(TSMAPI_FOUR.UI.NewElement("Text", "label")
				:SetStyle("textColor", "#e2e2e2")
				:SetStyle("fontHeight", 18)
				:SetText(L["Default Material Cost Method:"])
			)
		)
		:AddChild(TSMAPI_FOUR.UI.NewElement("Frame", "matCostMethod")
			:SetLayout("HORIZONTAL")
			:SetStyle("height", 32)
			:SetStyle("margin", { bottom = 16 })
			:AddChild(TSMAPI_FOUR.UI.NewElement("Input", "matCostInput")
				:SetText(TSM.db.global.craftingOptions.defaultMatCostMethod)
				:SetStyle("background", "#1ae2e2e2")
				:SetStyle("height", 32)
				:SetScript("OnEscapePressed", private.MatCostInputOnEscapePressed)
				:SetScript("OnEnterPressed", private.MatCostInputOnEnterPressed)
			)
			:AddChild(TSMAPI_FOUR.UI.NewElement("Button", "matCostReset")
				:SetStyle("borderTexture", "Interface\\Addons\\TradeSkillMaster\\Media\\ButtonEdgeFrame.blp")
				:SetStyle("borderSize", 7)
				:SetStyle("width", 120)
				:SetStyle("height", 29)
				:SetStyle("margin", { left = 20 })
				:SetText(RESET)
				:SetScript("OnClick", private.MatCostResetOnClick)
			)
		)
		:AddChild(TSMAPI_FOUR.UI.NewElement("Frame", "craftValue")
			:SetLayout("HORIZONTAL")
			:SetStyle("height", 26)
			:SetStyle("margin", { bottom = 8 })
			:AddChild(TSMAPI_FOUR.UI.NewElement("Text", "label")
				:SetStyle("textColor", "#e2e2e2")
				:SetStyle("fontHeight", 18)
				:SetStyle("margin", { right = 20 })
				:SetText(L["Default Craft Value Method:"])
			)
		)
		:AddChild(TSMAPI_FOUR.UI.NewElement("Frame", "craftValueMethod")
			:SetLayout("HORIZONTAL")
			:SetStyle("height", 32)
			:SetStyle("margin", { bottom = 16 })
			:AddChild(TSMAPI_FOUR.UI.NewElement("Input", "craftValueInput")
				:SetText(TSM.db.global.craftingOptions.defaultCraftPriceMethod)
				:SetStyle("background", "#1ae2e2e2")
				:SetStyle("height", 32)
				:SetScript("OnEscapePressed", private.CraftValueInputOnEscapePressed)
				:SetScript("OnEnterPressed", private.CraftValueInputOnEnterPressed)
			)
			:AddChild(TSMAPI_FOUR.UI.NewElement("Button", "craftValueReset")
				:SetStyle("borderTexture", "Interface\\Addons\\TradeSkillMaster\\Media\\ButtonEdgeFrame.blp")
				:SetStyle("borderSize", 7)
				:SetStyle("width", 120)
				:SetStyle("height", 29)
				:SetStyle("margin", { left = 20 })
				:SetText(RESET)
				:SetScript("OnClick", private.CraftValueResetOnClick)
			)
		)
		:AddChild(TSM.MainUI.Settings.CreateSettingLine("excludeCooldowns", L["Exclude crafts with cooldowns?"])
			:AddChild(TSMAPI_FOUR.UI.NewElement("Toggle", "cooldownToggle")
				:SetStyle("width", 80)
				:SetStyle("height", 20)
				:SetBooleanToggle(TSM.db.global.craftingOptions, "ignoreCDCraftCost")
			)
		)
end


-- ============================================================================
-- Helper Functions
-- ============================================================================

function private.ScaleInputOnEscapePressed(self)
	self:SetText(((TSMCraftingTradeSkillFrame and TSMCraftingTradeSkillFrame:GetFrameScale() or 1) * 100).."%")
	self:Draw()
end

function private.ScaleInputOnEnterPressed(self)
	local text = strtrim(self:GetText())
	local value = strmatch(text, "^([0-9]+) *%%?$")
	if value then
		value = TSMAPI_FOUR.Util.Round(value)
		if TSMCraftingTradeSkillFrame then
			TSMCraftingTradeSkillFrame:SetFrameScale(value / 100)
		end
	else
		value = (TSMCraftingTradeSkillFrame and TSMCraftingTradeSkillFrame:GetFrameScale() or 1) * 100
	end
	self:SetText(value .. "%")
	self:Draw()
end

function private.IgnoreCharactersOnSelectionChanged(self, selection)
	wipe(TSM.db.global.craftingOptions.ignoreCharacters)
	for character, selected in pairs(selection) do
		if selected then
			TSM.db.global.craftingOptions.ignoreCharacters[character] = true
		end
	end
end

function private.IgnoreGuildsOnSelectionChanged(self, selection)
	wipe(TSM.db.global.craftingOptions.ignoreGuilds)
	for guild, selected in pairs(selection) do
		if selected then
			TSM.db.global.craftingOptions.ignoreGuilds[guild] = true
		end
	end
end

function private.MatCostInputOnEscapePressed(self)
	self:SetText(TSM.db.global.craftingOptions.defaultMatCostMethod)
	self:Draw()
end

function private.MatCostInputOnEnterPressed(self)
	local value = self:GetText()
	local isValid, err = TSMAPI_FOUR.CustomPrice.Validate(value, "matprice")
	if isValid then
		TSM.db.global.craftingOptions.defaultMatCostMethod = value
		self:SetText(value)
		self:Draw()
	else
		TSM:Print(L["Invalid custom price."].." "..err)
		self:SetFocused(true)
	end
end

function private.MatCostResetOnClick(self)
	TSM.db.global.craftingOptions.defaultMatCostMethod = TSM.db:GetDefault("global", "craftingOptions", "defaultMatCostMethod")
	local input = self:GetElement("__parent.matCostInput")
	input:SetText(TSM.db.global.craftingOptions.defaultMatCostMethod)
	input:Draw()
end

function private.CraftValueInputOnEscapePressed(self)
	self:SetText(TSM.db.global.craftingOptions.defaultCraftPriceMethod)
	self:Draw()
end

function private.CraftValueInputOnEnterPressed(self)
	local value = self:GetText()
	local isValid, err = TSMAPI_FOUR.CustomPrice.Validate(value, "matprice")
	if isValid then
		TSM.db.global.craftingOptions.defaultCraftPriceMethod = value
		self:SetText(value)
		self:Draw()
	else
		TSM:Print(L["Invalid custom price."].." "..err)
		self:SetFocused(true)
	end
end

function private.CraftValueResetOnClick(self)
	TSM.db.global.craftingOptions.defaultCraftPriceMethod = TSM.db:GetDefault("global", "craftingOptions", "defaultCraftPriceMethod")
	local input = self:GetElement("__parent.craftValueInput")
	input:SetText(TSM.db.global.craftingOptions.defaultCraftPriceMethod)
	input:Draw()
end
